<?php

namespace App\Console\Commands;

use App\Models\Order;
use App\Services\DigiflazzService;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Log;

class DigiflazzSyncPending extends Command
{
    protected $signature = 'digiflazz:sync-pending {--limit=200}';
    protected $description = 'Sync status Digiflazz untuk order PAID yang masih Pending/Proses.';

    public function handle(DigiflazzService $digiflazz): int
    {
        $limit = max(1, (int) $this->option('limit'));

        $q = Order::query()
            ->where('payment_status', 'PAID')
            ->whereIn('buy_status', ['Pending', 'Proses'])
            ->whereNotNull('order_id')
            ->orderByDesc('id');

        $total = (int) $q->count();
        $this->info("Found {$total} pending/proses orders. Processing up to {$limit}...");

        $processed = 0;
        $failed = 0;

        $q->limit($limit)
            ->chunkById(50, function ($rows) use ($digiflazz, &$processed, &$failed) {
                foreach ($rows as $order) {
                    $processed++;

                    try {
                        $result = $digiflazz->syncStatus($order);

                        if (!($result['success'] ?? false)) {
                            $failed++;
                            Log::warning('[DIGIFLAZZ][CRON] sync failed', [
                                'order_id' => $order->order_id,
                                'message' => $result['message'] ?? null,
                            ]);
                        }
                    } catch (\Throwable $e) {
                        $failed++;
                        Log::warning('[DIGIFLAZZ][CRON] exception', [
                            'order_id' => $order->order_id,
                            'message' => $e->getMessage(),
                        ]);
                    }

                    usleep(200000);
                }
            });

        $this->info("Done. processed={$processed}, failed={$failed}");

        return Command::SUCCESS;
    }
}
