<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\LoginHistory;
use App\Models\User;
use App\Services\TurnstileService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Validator;
use Tymon\JWTAuth\Facades\JWTAuth;

class AuthController extends Controller
{
    private const DUPLICATE_REGISTER_MESSAGE = 'Akun Dengan Email/WhatsApp Tersebut Sudah Terdaftar, Silahkan Masuk Dengan Email Tersebut.';

    public function register(Request $request, TurnstileService $turnstile)
    {
        if (! $turnstile->verify($request->input('turnstile_token'), $request->ip())) {
            return response()->json([
                'success' => false,
                'message' => 'Verifikasi captcha gagal',
            ], 422);
        }

        $validator = Validator::make($request->all(), [
            'name' => ['required', 'string', 'min:2'],
            'email' => ['required', 'email'],
            'password' => ['required', 'string', 'min:6'],
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Data tidak valid',
                'errors' => $validator->errors(),
            ], 422);
        }

        $email = strtolower(trim((string) $request->email));
        $exists = User::query()->where('email', $email)->exists();
        if ($exists) {
            return response()->json([
                'success' => false,
                'message' => self::DUPLICATE_REGISTER_MESSAGE,
            ], 422);
        }

        $user = User::create([
            'name' => trim((string) $request->name),
            'email' => $email,
            'role' => 'basic',
            'saldo' => 0,
            'password' => Hash::make((string) $request->password),
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Register berhasil',
            'user' => [
                'id' => $user->id,
                'email' => $user->email,
                'name' => $user->name,
                'role' => $user->role,
                'saldo' => $user->saldo,
            ],
        ]);
    }

    public function login(Request $request, TurnstileService $turnstile)
    {
        if (! $turnstile->verify($request->input('turnstile_token'), $request->ip())) {
            return response()->json([
                'success' => false,
                'message' => 'Verifikasi captcha gagal',
            ], 422);
        }

        $validator = Validator::make($request->all(), [
            'email' => ['required', 'email'],
            'password' => ['required', 'string'],
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Data tidak valid',
                'errors' => $validator->errors(),
            ], 422);
        }

        $email = strtolower(trim((string) $request->email));
        $user = User::query()->where('email', $email)->first();

        if (! $user || ! Hash::check((string) $request->password, (string) $user->password)) {
            return response()->json([
                'success' => false,
                'message' => 'Email atau password salah',
            ], 401);
        }

        LoginHistory::create([
            'user_id' => $user->id,
            'provider' => 'credentials',
            'identifier' => $email,
            'ip' => (string) ($request->ip() ?? ''),
            'user_agent' => (string) ($request->userAgent() ?? ''),
            'logged_in_at' => now(),
        ]);

        $token = JWTAuth::fromUser($user);

        return response()->json([
            'success' => true,
            'message' => 'Login berhasil',
            'user' => [
                'id' => $user->id,
                'email' => $user->email,
                'name' => $user->name,
                'role' => $user->role,
                'saldo' => $user->saldo,
                'whatsapp' => $user->whatsapp,
            ],
            'token' => $token,
        ]);
    }

    public function googleAuth(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'id_token' => ['required', 'string'],
            'name' => ['nullable', 'string'],
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Data tidak valid',
                'errors' => $validator->errors(),
            ], 422);
        }

        $googleClientId = (string) (config('services.google.client_id') ?: env('GOOGLE_CLIENT_ID', ''));
        if ($googleClientId === '') {
            return response()->json([
                'success' => false,
                'message' => 'Google Client ID belum dikonfigurasi',
            ], 500);
        }

        $idToken = (string) $request->id_token;

        $res = Http::timeout(8)->get('https://oauth2.googleapis.com/tokeninfo', [
            'id_token' => $idToken,
        ]);

        if (! $res->ok()) {
            return response()->json([
                'success' => false,
                'message' => 'Token Google tidak valid',
            ], 401);
        }

        $payload = $res->json();
        $aud = (string) ($payload['aud'] ?? '');
        $email = strtolower(trim((string) ($payload['email'] ?? '')));
        $emailVerified = (string) ($payload['email_verified'] ?? 'false');

        if ($aud !== $googleClientId) {
            return response()->json([
                'success' => false,
                'message' => 'Token Google tidak sesuai aplikasi',
            ], 401);
        }

        if ($email === '' || $emailVerified !== 'true') {
            return response()->json([
                'success' => false,
                'message' => 'Email Google tidak valid',
            ], 401);
        }

        $name = trim((string) ($payload['name'] ?? $request->name ?? ''));

        $user = User::query()->where('email', $email)->first();
        if (! $user) {
            $user = User::create([
                'email' => $email,
                'name' => $name !== '' ? $name : $email,
                'role' => 'basic',
                'saldo' => 0,
                'password' => Hash::make(bin2hex(random_bytes(16))),
            ]);
        } else {
            if ($name !== '' && $user->name !== $name) {
                $user->update(['name' => $name]);
            }
        }

        $token = JWTAuth::fromUser($user);

        return response()->json([
            'success' => true,
            'message' => 'Login berhasil',
            'user' => [
                'id' => $user->id,
                'email' => $user->email,
                'name' => $user->name,
                'role' => $user->role,
                'saldo' => $user->saldo,
                'whatsapp' => $user->whatsapp,
            ],
            'token' => $token,
        ]);
    }
}