<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\LoginHistory;
use App\Models\OtpCode;
use App\Models\Setting;
use App\Models\User;
use App\Services\TurnstileService;
use App\Services\WhatsAppNotificationService;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\RateLimiter;
use Illuminate\Support\Facades\Validator;
use Tymon\JWTAuth\Facades\JWTAuth;

class OtpAuthController extends Controller
{
    private const DUPLICATE_REGISTER_MESSAGE = 'Akun Dengan Email/WhatsApp Tersebut Sudah Terdaftar, Silahkan Masuk Dengan Email Tersebut.';

    public function requestOtp(Request $request, TurnstileService $turnstile)
    {
        if (! $turnstile->verify($request->input('turnstile_token'), $request->ip())) {
            return response()->json([
                'success' => false,
                'message' => 'Verifikasi captcha gagal',
            ], 422);
        }

        $validator = Validator::make($request->all(), [
            'whatsapp' => ['required', 'string'],
            'purpose' => ['required', 'in:register,login,reset_password'],
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Data tidak valid',
                'errors' => $validator->errors(),
            ], 422);
        }

        $settings = Setting::pluck('value', 'key')->toArray();

        $whatsapp = $this->normalizeWhatsapp((string) $request->whatsapp);
        if (! $this->isValidWhatsapp($whatsapp)) {
            return response()->json([
                'success' => false,
                'message' => 'Nomor WhatsApp tidak valid',
            ], 422);
        }

        $purpose = (string) $request->purpose;

        if ($purpose === 'register') {
            if (User::query()->where('whatsapp', $whatsapp)->exists()) {
                return response()->json([
                    'success' => false,
                    'message' => self::DUPLICATE_REGISTER_MESSAGE,
                ], 422);
            }
        }

        if ($purpose === 'login' || $purpose === 'reset_password') {
            if (! User::query()->where('whatsapp', $whatsapp)->exists()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Akun belum terdaftar, silakan daftar terlebih dahulu',
                ], 404);
            }
        }

        $cooldownSeconds = (int) ($settings['otp.resend_cooldown_seconds'] ?? 60);
        $expiryMinutes = (int) ($settings['otp.expiry_minutes'] ?? 5);
        $otpLength = (int) ($settings['otp.length'] ?? 6);
        $maxAttempts = (int) ($settings['otp.max_attempts'] ?? 5);

        $ip = (string) ($request->ip() ?? '');
        $ua = (string) ($request->userAgent() ?? '');

        $rateKey = $this->rateKey($ip, $whatsapp, $purpose);
        if (RateLimiter::tooManyAttempts($rateKey, 5)) {
            $retryAfter = RateLimiter::availableIn($rateKey);
            return response()->json([
                'success' => false,
                'message' => 'Terlalu banyak permintaan, coba lagi nanti',
                'meta' => [
                    'cooldown_seconds' => (int) $retryAfter,
                ],
            ], 429);
        }

        RateLimiter::hit($rateKey, 60);

        $latest = OtpCode::query()
            ->where('whatsapp', $whatsapp)
            ->where('purpose', $purpose)
            ->whereNull('consumed_at')
            ->orderByDesc('id')
            ->first();

        if ($latest && $latest->resend_available_at && Carbon::parse($latest->resend_available_at)->isFuture()) {
            $remaining = Carbon::parse($latest->resend_available_at)->diffInSeconds(now());
            return response()->json([
                'success' => false,
                'message' => 'OTP masih dalam masa tunggu, silakan coba lagi nanti',
                'meta' => [
                    'cooldown_seconds' => (int) $remaining,
                ],
            ], 429);
        }

        $otpCode = $this->generateOtp($otpLength);

        $otp = OtpCode::create([
            'whatsapp' => $whatsapp,
            'purpose' => $purpose,
            'otp_hash' => Hash::make($otpCode),
            'expires_at' => now()->addMinutes($expiryMinutes),
            'resend_available_at' => now()->addSeconds($cooldownSeconds),
            'attempts' => 0,
            'max_attempts' => $maxAttempts,
            'ip' => $ip ?: null,
            'user_agent' => $ua ?: null,
            'consumed_at' => null,
        ]);

        $appName = (string) (($settings['general.title'] ?? null) ?: 'Website');
        $supportWa = (string) (($settings['sosmed.wa'] ?? null) ?: '');

        $vars = [
            'app_name' => $appName,
            'otp' => $otpCode,
            'otp_code' => $otpCode,
            'purpose' => $purpose,
            'expires_minutes' => (string) $expiryMinutes,
            'whatsapp' => $whatsapp,
            'date_time' => now()->format('d M Y, H:i') . ' WIB',
            'ip_address' => $ip,
            'device' => $ua,
            'support_whatsapp' => $supportWa,
        ];

        $templateKey = $purpose === 'register'
            ? 'otp_register'
            : ($purpose === 'reset_password' ? 'otp_reset_password' : 'otp_login');

        $fallback = $this->defaultOtpMessage($appName, $otpCode, $purpose, $expiryMinutes);

        app(WhatsAppNotificationService::class)->send($whatsapp, $templateKey, $vars, $fallback);

        return response()->json([
            'success' => true,
            'message' => 'OTP terkirim',
            'meta' => [
                'expires_in' => (int) now()->diffInSeconds($otp->expires_at),
                'cooldown_seconds' => $cooldownSeconds,
            ],
        ]);
    }

    public function verifyOtp(Request $request, TurnstileService $turnstile)
    {
        if (! $turnstile->verify($request->input('turnstile_token'), $request->ip())) {
            return response()->json([
                'success' => false,
                'message' => 'Verifikasi captcha gagal',
            ], 422);
        }

        $validator = Validator::make($request->all(), [
            'whatsapp' => ['required', 'string'],
            'otp' => ['required', 'string'],
            'purpose' => ['required', 'in:register,login,reset_password'],
            'name' => ['nullable', 'string'],
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Data tidak valid',
                'errors' => $validator->errors(),
            ], 422);
        }

        $whatsapp = $this->normalizeWhatsapp((string) $request->whatsapp);
        if (! $this->isValidWhatsapp($whatsapp)) {
            return response()->json([
                'success' => false,
                'message' => 'Nomor WhatsApp tidak valid',
            ], 422);
        }

        $purpose = (string) $request->purpose;

        $user = User::query()->where('whatsapp', $whatsapp)->first();

        if ($purpose === 'register' && $user) {
            return response()->json([
                'success' => false,
                'message' => self::DUPLICATE_REGISTER_MESSAGE,
            ], 422);
        }

        if (($purpose === 'login' || $purpose === 'reset_password') && ! $user) {
            return response()->json([
                'success' => false,
                'message' => 'Akun belum terdaftar, silakan daftar terlebih dahulu',
            ], 404);
        }

        $otpRow = OtpCode::query()
            ->where('whatsapp', $whatsapp)
            ->where('purpose', $purpose)
            ->whereNull('consumed_at')
            ->orderByDesc('id')
            ->first();

        if (! $otpRow) {
            return response()->json([
                'success' => false,
                'message' => 'OTP tidak ditemukan, silakan request OTP ulang',
            ], 404);
        }

        if (Carbon::parse($otpRow->expires_at)->isPast()) {
            return response()->json([
                'success' => false,
                'message' => 'OTP sudah kadaluarsa, silakan request OTP ulang',
            ], 422);
        }

        if ((int) $otpRow->attempts >= (int) $otpRow->max_attempts) {
            return response()->json([
                'success' => false,
                'message' => 'Percobaan OTP sudah melebihi batas',
            ], 429);
        }

        $otpInput = preg_replace('/\D+/', '', (string) $request->otp) ?? '';
        if ($otpInput === '' || ! Hash::check($otpInput, (string) $otpRow->otp_hash)) {
            $otpRow->increment('attempts');
            $remaining = max(0, ((int) $otpRow->max_attempts) - ((int) $otpRow->attempts));

            return response()->json([
                'success' => false,
                'message' => 'OTP salah',
                'meta' => [
                    'remaining_attempts' => $remaining,
                ],
            ], 422);
        }

        $otpRow->forceFill(['consumed_at' => now()])->save();

        if ($purpose === 'register') {
            $name = trim((string) ($request->name ?? ''));
            if ($name === '') {
                return response()->json([
                    'success' => false,
                    'message' => 'Nama wajib diisi',
                ], 422);
            }

            $user = User::create([
                'name' => $name,
                'whatsapp' => $whatsapp,
                'email' => $this->placeholderEmail($whatsapp),
                'role' => 'basic',
                'saldo' => 0,
                'password' => Hash::make(bin2hex(random_bytes(16))),
            ]);
        }

        LoginHistory::create([
            'user_id' => $user->id,
            'provider' => $purpose === 'reset_password' ? 'otp_reset_password' : 'otp',
            'identifier' => $whatsapp,
            'ip' => (string) ($request->ip() ?? ''),
            'user_agent' => (string) ($request->userAgent() ?? ''),
            'logged_in_at' => now(),
        ]);

        $token = JWTAuth::fromUser($user);

        return response()->json([
            'success' => true,
            'message' => 'Login berhasil',
            'user' => [
                'id' => $user->id,
                'email' => $user->email,
                'name' => $user->name,
                'role' => $user->role,
                'saldo' => $user->saldo,
                'whatsapp' => $user->whatsapp,
            ],
            'token' => $token,
        ]);
    }

    private function normalizeWhatsapp(string $value): string
    {
        $v = preg_replace('/\D+/', '', $value) ?? '';
        if (str_starts_with($v, '0')) {
            $v = '62' . substr($v, 1);
        }
        if (str_starts_with($v, '8')) {
            $v = '62' . $v;
        }
        return $v;
    }

    private function isValidWhatsapp(string $value): bool
    {
        if (! str_starts_with($value, '62')) {
            return false;
        }
        $len = strlen($value);
        return $len >= 10 && $len <= 16;
    }

    private function generateOtp(int $length): string
    {
        $length = max(4, min(8, $length));
        $min = (int) pow(10, $length - 1);
        $max = (int) pow(10, $length) - 1;
        return (string) random_int($min, $max);
    }

    private function rateKey(string $ip, string $whatsapp, string $purpose): string
    {
        return 'otp:request:' . sha1($ip . '|' . $whatsapp . '|' . $purpose);
    }

    private function placeholderEmail(string $whatsapp): string
    {
        return $whatsapp . '@whatsapp.local';
    }

    private function defaultOtpMessage(string $appName, string $otp, string $purpose, int $expiryMinutes): string
    {
        $title = $purpose === 'register'
            ? 'OTP Daftar'
            : ($purpose === 'reset_password' ? 'OTP Reset Password' : 'OTP Login');

        return "*{$appName} - {$title}*\n\nKode OTP kamu: *{$otp}*\nBerlaku selama *{$expiryMinutes} menit*.\n\nJangan bagikan kode ini ke siapa pun.";
    }
}