<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Setting;
use App\Models\User;
use App\Models\WhatsappPasswordResetToken;
use App\Services\WhatsAppNotificationService;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Password;
use Illuminate\Support\Facades\RateLimiter;
use Illuminate\Support\Facades\Validator;

class PasswordResetController extends Controller
{
    public function forgot(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'method' => ['required', 'in:email,whatsapp'],
            'email' => ['nullable', 'email'],
            'whatsapp' => ['nullable', 'string'],
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Data tidak valid',
                'errors' => $validator->errors(),
            ], 422);
        }

        $method = (string) $request->method;

        $ip = (string) ($request->ip() ?? '');
        $ua = (string) ($request->userAgent() ?? '');
        $rateKey = 'password:forgot:' . sha1($ip . '|' . $method);

        if (RateLimiter::tooManyAttempts($rateKey, 10)) {
            $retryAfter = RateLimiter::availableIn($rateKey);
            return response()->json([
                'success' => false,
                'message' => 'Terlalu banyak permintaan, coba lagi nanti',
                'meta' => [
                    'cooldown_seconds' => (int) $retryAfter,
                ],
            ], 429);
        }

        RateLimiter::hit($rateKey, 60);

        if ($method === 'whatsapp') {
            return response()->json([
                'success' => false,
                'message' => 'Gunakan verifikasi OTP untuk reset password via WhatsApp',
            ], 422);
        }

        $email = strtolower(trim((string) $request->email));
        if ($email === '') {
            return response()->json([
                'success' => false,
                'message' => 'Email wajib diisi',
            ], 422);
        }

        Password::sendResetLink(['email' => $email]);

        return response()->json([
            'success' => true,
            'message' => 'Jika email terdaftar, link reset password akan dikirim',
        ]);
    }

    public function reset(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'token' => ['required', 'string'],
            'password' => ['required', 'string', 'min:6'],
            'password_confirmation' => ['required', 'same:password'],
            'email' => ['nullable', 'email'],
            'whatsapp' => ['nullable', 'string'],
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Data tidak valid',
                'errors' => $validator->errors(),
            ], 422);
        }

        $ip = (string) ($request->ip() ?? '');
        $ua = (string) ($request->userAgent() ?? '');
        $rateKey = 'password:reset:' . sha1($ip);

        if (RateLimiter::tooManyAttempts($rateKey, 15)) {
            $retryAfter = RateLimiter::availableIn($rateKey);
            return response()->json([
                'success' => false,
                'message' => 'Terlalu banyak percobaan, coba lagi nanti',
                'meta' => [
                    'cooldown_seconds' => (int) $retryAfter,
                ],
            ], 429);
        }

        RateLimiter::hit($rateKey, 60);

        $token = (string) $request->token;

        $email = strtolower(trim((string) ($request->email ?? '')));
        $whatsapp = trim((string) ($request->whatsapp ?? ''));

        if ($whatsapp !== '' && $email === '') {
            return $this->resetByWhatsapp($request, $whatsapp, $token, $ip, $ua);
        }

        if ($email === '') {
            return response()->json([
                'success' => false,
                'message' => 'Email atau WhatsApp wajib diisi',
            ], 422);
        }

        $status = Password::reset(
            [
                'email' => $email,
                'token' => $token,
                'password' => (string) $request->password,
                'password_confirmation' => (string) $request->password_confirmation,
            ],
            function (User $user, string $password) {
                $user->forceFill([
                    'password' => Hash::make($password),
                ])->save();
            }
        );

        if ($status !== Password::PASSWORD_RESET) {
            return response()->json([
                'success' => false,
                'message' => 'Token tidak valid atau sudah kadaluarsa',
            ], 422);
        }

        return response()->json([
            'success' => true,
            'message' => 'Password berhasil diperbarui',
        ]);
    }

    private function resetByWhatsapp(Request $request, string $whatsapp, string $token, string $ip, string $ua)
    {
        $normalized = $this->normalizeWhatsapp($whatsapp);

        $row = WhatsappPasswordResetToken::query()
            ->where('whatsapp', $normalized)
            ->whereNull('consumed_at')
            ->orderByDesc('id')
            ->first();

        if (! $row) {
            return response()->json([
                'success' => false,
                'message' => 'Token tidak valid atau sudah kadaluarsa',
            ], 422);
        }

        if ($row->expires_at && Carbon::parse($row->expires_at)->isPast()) {
            return response()->json([
                'success' => false,
                'message' => 'Token tidak valid atau sudah kadaluarsa',
            ], 422);
        }

        $tokenHash = hash('sha256', $token);
        if (! hash_equals((string) $row->token_hash, (string) $tokenHash)) {
            return response()->json([
                'success' => false,
                'message' => 'Token tidak valid atau sudah kadaluarsa',
            ], 422);
        }

        $user = User::where('whatsapp', $normalized)->first();
        if (! $user) {
            return response()->json([
                'success' => false,
                'message' => 'Akun tidak ditemukan',
            ], 404);
        }

        DB::transaction(function () use ($row, $request, $user) {
            $fresh = WhatsappPasswordResetToken::lockForUpdate()->find($row->id);

            if (! $fresh || $fresh->consumed_at) {
                abort(422, 'Token tidak valid atau sudah kadaluarsa');
            }

            $fresh->forceFill([
                'consumed_at' => now(),
            ])->save();

            $user->forceFill([
                'password' => Hash::make((string) $request->password),
            ])->save();
        });

        return response()->json([
            'success' => true,
            'message' => 'Password berhasil diperbarui',
        ]);
    }

    private function normalizeWhatsapp(string $value): string
    {
        $v = preg_replace('/\D+/', '', $value) ?? '';
        if (str_starts_with($v, '0')) {
            $v = '62' . substr($v, 1);
        }
        if (str_starts_with($v, '8')) {
            $v = '62' . $v;
        }
        return $v;
    }
}