<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Order;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Http\Request;

class TransactionController extends Controller
{
    private function normalizeWhatsapp(?string $value): ?string
    {
        if (!$value) return null;
        $v = preg_replace('/[^0-9]/', '', $value);
        if ($v === '') return null;
        if (str_starts_with($v, '0')) $v = '62' . substr($v, 1);
        if (!str_starts_with($v, '62')) $v = '62' . $v;
        return $v;
    }

    private function baseQueryForUser(Request $request): Builder
    {
        $user = $request->user();
        $userId = $user?->id;
        $wa = $this->normalizeWhatsapp($user?->whatsapp ?? null);

        if (!$userId && !$wa) {
            abort(401, 'Unauthorized');
        }

        return Order::query()->where(function ($q) use ($userId, $wa) {
            if ($userId && $wa) {
                $q->where('user_id', $userId)->orWhere('whatsapp', $wa);
                return;
            }

            if ($userId) {
                $q->where('user_id', $userId);
                return;
            }

            if ($wa) {
                $q->where('whatsapp', $wa);
            }
        });
    }

    private function normalizeBuyStatus(?string $buyStatus, ?string $paymentStatus = null): string
    {
        $p = strtoupper(trim((string) $paymentStatus));
        if ($p === 'PAID') return 'Sukses';
        if ($p === 'UNPAID') return 'Pending';
        if ($p === 'EXPIRED') return 'Batal';
        if ($p === 'FAILED') return 'Gagal';

        $v = trim((string) $buyStatus);
        $vLower = mb_strtolower($v);

        if (in_array($vLower, ['success', 'sukses', 'berhasil'], true)) return 'Sukses';
        if (in_array($vLower, ['pending', 'menunggu pembayaran'], true)) return 'Pending';
        if (in_array($vLower, ['batal', 'expired', 'kedaluwarsa'], true)) return 'Batal';
        if (in_array($vLower, ['gagal', 'failed', 'error'], true)) return 'Gagal';

        if ($vLower !== '') {
            return mb_strtoupper(mb_substr($vLower, 0, 1)) . mb_substr($vLower, 1);
        }

        return 'Pending';
    }

    private function mapOrder(Order $o): array
    {
        return [
            'id' => $o->id,
            'order_id' => $o->order_id,
            'amount' => (int) ($o->total_price ?? 0),
            'total_price' => (int) ($o->total_price ?? 0),
            'payment_status' => $o->payment_status,
            'buy_status' => $this->normalizeBuyStatus($o->buy_status ?? null, $o->payment_status ?? null),
            'status' => $this->normalizeBuyStatus($o->buy_status ?? null, $o->payment_status ?? null),
            'created_at' => $o->created_at?->toIso8601String(),
        ];
    }

    public function index(Request $request)
    {
        $q = trim((string) $request->query('q', ''));
        $perPage = (int) $request->query('per_page', 10);
        if ($perPage < 1) $perPage = 10;
        if ($perPage > 100) $perPage = 100;

        $query = $this->baseQueryForUser($request)->orderByDesc('id');

        if ($q !== '') {
            $query->where('order_id', 'like', '%' . $q . '%');
        }

        $paged = $query->paginate($perPage);

        $rows = collect($paged->items())
            ->map(fn (Order $o) => $this->mapOrder($o))
            ->values();

        return response()->json([
            'success' => true,
            'data' => $rows,
            'meta' => [
                'current_page' => $paged->currentPage(),
                'per_page' => $paged->perPage(),
                'total' => $paged->total(),
                'last_page' => $paged->lastPage(),
            ],
        ]);
    }

    public function show(Request $request, string $order_id)
    {
        $order = $this->baseQueryForUser($request)
            ->where('order_id', $order_id)
            ->first();

        if (!$order) {
            return response()->json([
                'success' => false,
                'message' => 'Transaksi tidak ditemukan',
            ], 404);
        }

        return response()->json([
            'success' => true,
            'data' => $this->mapOrder($order),
        ]);
    }

    public function summary(Request $request)
    {
        $base = $this->baseQueryForUser($request);

        $total = (int) (clone $base)->count();
        $paid = (int) (clone $base)->whereRaw('UPPER(payment_status) = ?', ['PAID'])->count();
        $unpaid = (int) (clone $base)->whereRaw('UPPER(payment_status) = ?', ['UNPAID'])->count();
        $expired = (int) (clone $base)->whereRaw('UPPER(payment_status) = ?', ['EXPIRED'])->count();

        $failed = (int) (clone $base)->where(function ($q) {
            $q->whereRaw('UPPER(payment_status) = ?', ['FAILED'])
              ->orWhereRaw('LOWER(buy_status) LIKE ?', ['%gagal%']);
        })->count();

        $failedExpired = (int) ($failed + $expired);

        return response()->json([
            'success' => true,
            'data' => [
                'total' => $total,
                'paid' => $paid,
                'unpaid' => $unpaid,
                'failed' => $failed,
                'expired' => $expired,
                'failed_expired' => $failedExpired,
            ],
        ]);
    }

    public function transactionSummary(Request $request)
    {
        return $this->summary($request);
    }
}