<?php

namespace App\Http\Controllers;

use App\Events\PaymentStatusUpdated;
use App\Events\RealtimeTransaction;
use App\Models\Account;
use App\Models\Order;
use App\Models\Product;
use App\Services\DigiflazzService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class PaydisiniCallbackController extends Controller
{
    private ?string $apiKey = null;

    private array $allowedIps = [
        '45.87.242.188',
    ];

    public function __construct()
    {
        $this->apiKey = (string) config('services.paydisini.api_key');
    }

    public function handle(Request $request)
    {
        if (! $this->isValidIp($request->ip())) {
            Log::warning("Unauthorized Paydisini callback attempt from IP: " . $request->ip());
            return response()->json(['success' => false, 'message' => 'Unauthorized IP'], 403);
        }

        $data = $request->all();

        $key = $data['key'] ?? null;
        $payId = $data['pay_id'] ?? null;
        $uniqueCode = $data['unique_code'] ?? null;
        $status = $data['status'] ?? null;
        $signature = $data['signature'] ?? null;

        if (! $key || ! $payId || ! $uniqueCode || ! $status || ! $signature) {
            return response()->json(['success' => false, 'message' => 'Missing parameters'], 400);
        }

        $expectedSignature = md5($this->apiKey . $uniqueCode . 'CallbackStatus');

        if ($signature !== $expectedSignature) {
            return response()->json(['success' => false, 'message' => 'Invalid signature'], 401);
        }

        $order = Order::where('order_id', $uniqueCode)
            ->where('payment_status', 'UNPAID')
            ->first();

        if (! $order) {
            return response()->json(['success' => false, 'message' => "No order found: $uniqueCode"], 404);
        }

        $st = strtolower((string) $status);

        if ($st === 'success') {
            $order->update([
                'payment_status' => 'PAID',
                'buy_status' => 'Proses',
            ]);

            if ($order->type === 'Akun') {
                $this->processAccountTransaction($order);
            } else {
                $this->processDigiflazzTransaction($order);
            }

            $order->refresh();

            event(new PaymentStatusUpdated($order));
            event(new RealtimeTransaction($order));

            return response()->json(['success' => true]);
        }

        if ($st === 'canceled') {
            $order->update([
                'payment_status' => 'EXPIRED',
                'buy_status' => 'Batal',
            ]);

            event(new PaymentStatusUpdated($order));
            event(new RealtimeTransaction($order));

            return response()->json(['success' => true]);
        }

        $order->update([
            'payment_status' => 'FAILED',
            'buy_status' => 'Batal',
        ]);

        event(new PaymentStatusUpdated($order));
        event(new RealtimeTransaction($order));

        return response()->json(['success' => true]);
    }

    private function processDigiflazzTransaction(Order $order): void
    {
        app(DigiflazzService::class)->submit($order);
    }

    private function processAccountTransaction(Order $order): void
    {
        $product = Product::where('code', $order->code_product)->first();

        if (! $product) {
            $order->update([
                'buy_status' => 'Gagal',
                'df_message' => 'Product not found',
            ]);

            return;
        }

        if ($product->type_transaction === 'manual') {
            $manualMessage = "Pesanan Anda sedang diproses secara manual.\nAdmin akan segera menghubungi Anda.\nHarap tunggu dan pastikan nomor WhatsApp Anda aktif.";

            $order->update([
                'serial_number' => $manualMessage,
                'buy_status' => 'Proses',
                'df_message' => 'Order will be processed manually',
            ]);

            return;
        }

        $account = Account::where('code_account', $order->code_product)
            ->where('stok', '>', 0)
            ->where('status', 'available')
            ->first();

        if (! $account) {
            $order->update([
                'buy_status' => 'Gagal',
                'df_message' => 'Account not available',
                'serial_number' => 'Stok akun tidak tersedia, hubungi admin untuk mendapatkan bantuan!',
            ]);

            return;
        }

        $credentials = "Berikut detail akun Anda:\nEmail: {$account->email}\nPassword: {$account->password}\n";

        if ($account->descriptions) {
            $credentials .= "Keterangan: {$account->descriptions}\n";
        }

        $order->update([
            'serial_number' => $credentials,
            'buy_status' => 'Sukses',
            'df_message' => 'Account delivered successfully',
        ]);

        $account->decrement('stok');

        if ($account->stok <= 0) {
            $account->update(['status' => 'sold']);
        }
    }

    private function isValidIp(string $ip): bool
    {
        return in_array($ip, $this->allowedIps, true);
    }
}