<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use App\Models\PaymentMethod;
use App\Models\Setting;

class PaymentChannelController extends Controller
{
    public function fetchChannels($provider = 'tripay')
    {
        if ($provider === 'tripay') {
            return $this->fetchFromTripay();
        } elseif ($provider === 'paydisini') {
            return $this->fetchFromPaydisini();
        } else {
            return response()->json([
                'success' => false,
                'message' => 'Provider tidak dikenal.'
            ], 400);
        }
    }

    public function fetchTripayChannels()
    {
        return $this->fetchFromTripay();
    }

    public function fetchPaydisiniChannels()
    {
        return $this->fetchFromPaydisini();
    }

    protected function fetchFromTripay()
    {
        $settings = Setting::all()->pluck('value', 'key');

        $apiKey = $settings['tripay.apikey'] ?? null;

        $response = Http::withHeaders([
            'Authorization' => 'Bearer ' . $apiKey,
        ])->get('https://tripay.co.id/api/merchant/payment-channel');

        if ($response->successful()) {
            $channels = $response->json('data') ?? [];

            foreach ($channels as $channel) {
                PaymentMethod::updateOrCreate(
                    ['code' => $channel['code']],
                    [
                        'images' => $channel['icon_url'] ?? null,
                        'name' => $channel['name'],
                        'group' => $channel['group'],
                        'type' => $channel['type'],
                        'fee' => (int) ($channel['total_fee']['flat'] ?? 0),
                        'fee_percent' => (float) ($channel['total_fee']['percent'] ?? 0),
                        'minimum_amount' => (int) ($channel['minimum_amount'] ?? 0),
                        'maximum_amount' => isset($channel['maximum_amount']) ? (int) $channel['maximum_amount'] : null,
                        'provider' => 'tripay',
                        'status' => !empty($channel['active']) ? 1 : 0,
                    ]
                );
            }

            return response()->json([
                'success' => true,
                'message' => 'Tripay payment channels updated successfully.'
            ]);
        }

        $errorMessage = $response->json('message') ?? 'Unknown error';
        Log::error('Tripay fetch error: ' . $errorMessage);

        return response()->json([
            'success' => false,
            'message' => $errorMessage
        ], 400);
    }

    protected function fetchFromPaydisini()
    {
        $settings = Setting::all()->pluck('value', 'key');

        $apiKey = $settings['paydisini.apikey'] ?? null;
        $signature = md5($apiKey . 'PaymentChannel');

        $response = Http::asForm()->post('https://api.paydisini.co.id/v1/', [
            'key' => $apiKey,
            'request' => 'payment_channel',
            'signature' => $signature,
        ]);

        if ($response->successful() && $response->json('success')) {
            $channels = $response->json('data') ?? [];

            foreach ($channels as $channel) {
                $feeRaw = $channel['fee'];
                $feeFlat = 0;
                $feePercent = 0;

                if (is_string($feeRaw) && str_contains($feeRaw, '%')) {
                    $feePercent = floatval(str_replace('%', '', $feeRaw));
                } else {
                    $feeFlat = floatval($feeRaw);
                }

                PaymentMethod::updateOrCreate(
                    [
                        'code' => $channel['id'],
                        'provider' => 'paydisini',
                    ],
                    [
                        'name' => $channel['name'],
                        'type' => $channel['type'],
                        'group' => $channel['type'],
                        'images' => 'default.png',
                        'fee' => $feeFlat,
                        'fee_percent' => $feePercent,
                        'minimum_amount' => $channel['minimum'],
                        'maximum_amount' => $channel['maximum'],
                        'status' => $channel['status'] === 'ON' ? 1 : 0,
                    ]
                );
            }

            return response()->json([
                'success' => true,
                'message' => 'Payment channels from PayDisini updated successfully.'
            ]);
        }

        Log::error('Failed fetching PayDisini channels: ' . $response->body());

        return response()->json([
            'success' => false,
            'message' => 'Failed to fetch from PayDisini'
        ], 400);
    }
}