<?php

namespace App\Http\Controllers;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use App\Models\Game;
use App\Models\Product;
use App\Models\ProductCategory;
use App\Models\Profit;
use App\Models\Setting;
use App\Models\Category;

class ProductDigiflazzController extends Controller
{
    private string $siteName = 'Website';

    private function ensureCategoryLogo(string $categoryTitle, string $categoryGame): string
    {
        $slug = Str::slug($categoryGame . '-' . $categoryTitle);
        $path = "product/logo/{$slug}.svg";

        if (!Storage::disk('public')->exists($path)) {
            $title = mb_strtoupper(mb_substr($categoryTitle, 0, 18));
            $subtitle = mb_strtoupper(mb_substr($categoryGame, 0, 18));
            $svg = $this->buildSvg(256, 256, $title, $subtitle);
            Storage::disk('public')->put($path, $svg);
        }

        return $path;
    }

    private function ensureProductBanner(string $productName, string $brand, string $code): string
    {
        $slug = Str::slug(($brand !== '' ? $brand : $productName) . '-' . $code);
        $path = "product/banner/{$slug}.svg";

        if (!Storage::disk('public')->exists($path)) {
            $title = mb_strtoupper(mb_substr($brand !== '' ? $brand : $productName, 0, 28));
            $subtitle = mb_substr($productName, 0, 40);
            $svg = $this->buildSvg(1200, 400, $title, $subtitle);
            Storage::disk('public')->put($path, $svg);
        }

        return $path;
    }

    private function buildSvg(int $width, int $height, string $title, string $subtitle): string
    {
        $title = htmlspecialchars($title, ENT_QUOTES, 'UTF-8');
        $subtitle = htmlspecialchars($subtitle, ENT_QUOTES, 'UTF-8');
        $w = (string) $width;
        $h = (string) $height;

        return "<svg xmlns=\"http://www.w3.org/2000/svg\" width=\"{$w}\" height=\"{$h}\" viewBox=\"0 0 {$w} {$h}\"><defs><linearGradient id=\"g\" x1=\"0\" y1=\"0\" x2=\"1\" y2=\"1\"><stop offset=\"0\" stop-color=\"#0f172a\"/><stop offset=\"1\" stop-color=\"#1f2937\"/></linearGradient></defs><rect width=\"100%\" height=\"100%\" fill=\"url(#g)\"/><rect x=\"24\" y=\"24\" width=\"" . ($width - 48) . "\" height=\"" . ($height - 48) . "\" rx=\"24\" fill=\"rgba(255,255,255,0.06)\"/><text x=\"50%\" y=\"46%\" text-anchor=\"middle\" font-family=\"Inter,system-ui,-apple-system,Segoe UI,Roboto,Arial\" font-size=\"" . (int) round($height * 0.12) . "\" fill=\"#e5e7eb\" font-weight=\"700\">{$title}</text><text x=\"50%\" y=\"60%\" text-anchor=\"middle\" font-family=\"Inter,system-ui,-apple-system,Segoe UI,Roboto,Arial\" font-size=\"" . (int) round($height * 0.06) . "\" fill=\"#cbd5e1\" font-weight=\"500\">{$subtitle}</text></svg>";
    }

    private function normalizeDigiflazzMainCategory(?string $raw): string
    {
        $value = trim((string) $raw);
        if ($value === '') {
            return 'Pulsa';
        }

        $key = mb_strtolower($value);

        if (str_contains($key, 'lainnya')) return 'Pulsa';
        if (str_contains($key, 'game')) return 'Games';
        if (str_contains($key, 'pulsa')) return 'Pulsa';
        if (str_contains($key, 'data')) return 'Data';
        if (str_contains($key, 'voucher')) return 'Voucher';
        if (str_contains($key, 'pln')) return 'PLN';

        if (
            str_contains($key, 'e-money') ||
            str_contains($key, 'emoney') ||
            str_contains($key, 'e money') ||
            str_contains($key, 'ewallet') ||
            str_contains($key, 'e-wallet') ||
            str_contains($key, 'e wallet')
        ) return 'E-Money';

        if (str_contains($key, 'telkom')) return 'Telkom';
        if (str_contains($key, 'tv')) return 'TV';
        if (str_contains($key, 'bpjs')) return 'BPJS';
        if (str_contains($key, 'pdam')) return 'PDAM';
        if (str_contains($key, 'gas')) return 'Gas';
        if (str_contains($key, 'pajak')) return 'Pajak';
        if (str_contains($key, 'sms')) return 'SMS';

        return 'Pulsa';
    }

    private function ensureMainCategory(string $title): Category
    {
        return Category::firstOrCreate(
            ['title' => $title],
            ['sort' => 0]
        );
    }

    private function ensureGameImage(string $key): string
    {
        $slug = Str::slug($key);
        $path = "games/image/{$slug}.svg";

        if (!Storage::disk('public')->exists($path)) {
            $title = mb_strtoupper(mb_substr($key, 0, 18));
            $svg = $this->buildSvg(512, 512, $title, 'TOPUP');
            Storage::disk('public')->put($path, $svg);
        }

        return $path;
    }

    private function ensureGameBanner(string $key): string
    {
        $slug = Str::slug($key);
        $path = "games/banner/{$slug}.svg";

        if (!Storage::disk('public')->exists($path)) {
            $title = mb_strtoupper(mb_substr($key, 0, 28));
            $svg = $this->buildSvg(1600, 600, $title, 'TOPUP GAME');
            Storage::disk('public')->put($path, $svg);
        }

        return $path;
    }

    private function smartTitle(string $text): string
    {
        $t = trim($text);
        if ($t === '') return '';

        $upper = mb_strtoupper($t);
        if ($t !== $upper) return $t;

        $lower = mb_strtolower($t);
        $title = mb_convert_case($lower, MB_CASE_TITLE, 'UTF-8');

        $replacements = [
            'Pln' => 'PLN',
            'Bpjs' => 'BPJS',
            'Pdam' => 'PDAM',
            'Tv' => 'TV',
            'Sms' => 'SMS',
            'Pubg' => 'PUBG',
            'Ml' => 'ML',
            'Uid' => 'UID',
            'Id' => 'ID',
            'Ios' => 'iOS',
            'Android' => 'Android',
        ];

        foreach ($replacements as $from => $to) {
            $title = preg_replace('/\b' . preg_quote($from, '/') . '\b/u', $to, $title) ?? $title;
        }

        return $title;
    }

    private function gameKey(string $value): string
    {
        $v = mb_strtolower(trim($value));
        $v = preg_replace('/[^a-z0-9]+/iu', '', $v) ?? '';
        return $v;
    }

    private function gameTitleFromDb(string $brandOrTitle): string
    {
        static $map = null;

        if ($map === null) {
            $map = [];
            $rows = Game::query()->select(['brand', 'title'])->get();

            foreach ($rows as $row) {
                $brand = trim((string) ($row->brand ?? ''));
                $title = trim((string) ($row->title ?? ''));

                if ($brand === '') continue;

                $k = $this->gameKey($brand);
                if ($k === '') continue;

                $map[$k] = $title !== '' ? $title : $brand;
            }
        }

        $k = $this->gameKey($brandOrTitle);
        if ($k !== '' && isset($map[$k])) {
            return (string) $map[$k];
        }

        return trim($brandOrTitle) !== '' ? trim($brandOrTitle) : 'Produk';
    }

    private function developerFor(string $brand, string $title): string
    {
        $dbTitle = $this->gameTitleFromDb($brand !== '' ? $brand : $title);
        $key = $this->gameKey($dbTitle);

        $map = [
            'arenaofvalor' => 'TiMi Studio Group - Tencent Games',
            'amongus' => 'Innersloth',
            'axis' => 'XL Axiata',
            'bloodstrike' => 'NetEase Games',
            'byu' => 'Telkomsel',
            'callofdutymobile' => 'TiMi Studio Group - Tencent Games / Activision',
            'dana' => 'DANA Indonesia',
            'deltaforce' => 'Team Jade - TiMi Studio Group',
            'fcmobile' => 'EA SPORTS',
            'freefire' => 'Garena',
            'freefiremax' => 'Garena',
            'genshinimpact' => 'HoYoverse',
            'gopay' => 'GoTo Financial',
            'googleplayindonesia' => 'Google',
            'honkaiimpact3' => 'HoYoverse',
            'honorofkings' => 'TiMi Studio Group - Tencent Games',
            'honkaistarrail' => 'HoYoverse',
            'indriver' => 'inDrive',
            'indosat' => 'Indosat Ooredoo Hutchison',
            'isaku' => 'i.saku',
            'itunes' => 'Apple',
            'lifeaftercredits' => 'NetEase Games',
            'linkaja' => 'LinkAja',
            'lordsmobile' => 'IGG',
            'magicchess' => 'MOONTON',
            'mobilelegends' => 'MOONTON',
            'ovo' => 'OVO',
            'pln' => 'PLN',
            'pubgmobile' => 'LightSpeed Studios - Tencent Games',
            'ragnarokmclassic' => 'Gravity',
            'stumbleguys' => 'Scopely',
            'smartfren' => 'Smartfren',
            'shopeepay' => 'Shopee',
            'steamwalletidr' => 'Valve',
            'tri' => 'Indosat Ooredoo Hutchison',
            'telkomsel' => 'Telkomsel',
            'valorant' => 'Riot Games',
            'genflix' => 'Genflix',
            'garena' => 'Garena',
            'viu' => 'PCCW',
            'riotcash' => 'Riot Games',
            'razergold' => 'Razer',
            'wetv' => 'Tencent Video',
            'xl' => 'XL Axiata',
            'pointblank' => 'Zepetto',
        ];

        if (isset($map[$key])) {
            return $map[$key];
        }

        return $brand !== '' ? $brand : ($dbTitle !== '' ? $dbTitle : $title);
    }

    private function descriptionFor(string $name): string
    {
        $game = trim($name) !== '' ? trim($name) : 'Produk';
        $web = trim((string) $this->siteName) !== '' ? trim((string) $this->siteName) : 'Website';

        $gameEsc = htmlspecialchars($game, ENT_QUOTES, 'UTF-8');
        $webEsc = htmlspecialchars($web, ENT_QUOTES, 'UTF-8');

        return
            "TopUp <strong>{$gameEsc}</strong> harga paling murah, aman, cepat, dan terpercaya hanya di <strong>{$webEsc}</strong>.<br><br>" .
            "<h4 style=\"font-size:1.1rem\"><strong>Cara TopUp <strong>{$gameEsc}</strong> :</strong></h4>" .
            "<ol>" .
            "<li>Pilih Nominal</li>" .
            "<li>Masukkan Data Akun</li>" .
            "<li>Pilih Produk</li>" .
            "<li>Pilih Pembayaran</li>" .
            "<li>Isi Detail Kontak</li>" .
            "<li>Klik Pesan Sekarang dan lakukan Pembayaran</li>" .
            "<li>Selesai</li>" .
            "</ol>";
    }

    private function stripLeadingGameName(string $productName, string $gameName, string $brand): string
    {
        $original = trim($productName);
        if ($original === '') return '';

        $alnum = fn (string $s): string => preg_replace('/[^a-z0-9]+/iu', '', $s) ?? '';

        $abbr = function (string $s): string {
            $s = trim($s);
            if ($s === '') return '';
            $words = preg_split('/\s+/u', $s) ?: [];
            $letters = '';
            foreach ($words as $w) {
                $w = trim($w);
                if ($w === '') continue;
                $letters .= mb_substr($w, 0, 1);
            }
            return mb_strtoupper($letters);
        };

        $dropTrailingS = function (string $s): string {
            $s = trim($s);
            if ($s === '') return '';
            $last = mb_substr($s, -1);
            if ($last === 's' || $last === 'S') {
                return rtrim(mb_substr($s, 0, mb_strlen($s) - 1));
            }
            return $s;
        };

        $baseCandidates = array_values(array_unique(array_filter([
            trim($brand),
            trim($gameName),
            trim($this->smartTitle($brand)),
            trim($this->smartTitle($gameName)),
        ])));

        $candidates = [];
        foreach ($baseCandidates as $c) {
            $c1 = trim($c);
            $c2 = $dropTrailingS($c1);

            $a1 = $alnum($c1);
            $a2 = $dropTrailingS($a1);

            $b1 = $abbr($c1);
            $b2 = $dropTrailingS($b1);

            $candidates[] = $c1;
            $candidates[] = $c2;
            $candidates[] = $a1;
            $candidates[] = $a2;
            $candidates[] = $b1;
            $candidates[] = $b2;
        }

        $candidates = array_values(array_unique(array_filter($candidates)));

        $separators = '(?:\s*[-–—|:]\s*|\s+)';
        $patterns = [];

        foreach ($candidates as $c) {
            $patterns[] = preg_quote($c, '/');
        }

        $pattern = '/^(?:' . implode('|', $patterns) . ')(?:' . $separators . ')?/iu';

        $cleaned = preg_replace($pattern, '', $original);
        $cleaned = trim((string) $cleaned);

        return $cleaned !== '' ? $cleaned : $original;
    }

    private function isPointBlank(string $value): bool
    {
        $v = mb_strtolower(trim($value));
        if ($v === '') return false;
        $v = preg_replace('/\s+/u', ' ', $v) ?? $v;
        return $v === 'point blank' || str_contains($v, 'point blank');
    }

    private function isVoucherProductName(string $productName): bool
    {
        return preg_match('/\bvoucher\b/iu', (string) $productName) === 1;
    }

    private function ensureGameFromDigiflazz(string $brand, string $title, Category $category): ?Game
    {
        $brand = trim($brand);
        $title = trim($title);
        if ($brand === '') return null;

        $cleanTitle = $this->smartTitle($title !== '' ? $title : $brand);
        $developers = $this->developerFor($brand, $cleanTitle);
        $desc = $this->descriptionFor($cleanTitle !== '' ? $cleanTitle : $brand);

        $existing = Game::where('brand', $brand)->first();

        if ($existing) {
            $update = [
                'title' => $cleanTitle !== '' ? $cleanTitle : $brand,
                'developers' => $developers !== '' ? $developers : ($brand !== '' ? $brand : 'Developer'),
                'category_id' => (string) $category->id,
                'description' => $desc,
                'status' => true,
            ];

            if ((string) ($existing->slug ?? '') === '') {
                $update['slug'] = Str::slug($brand);
            }

            if ((string) ($existing->image ?? '') === '') {
                $update['image'] = $this->ensureGameImage($brand);
            }

            if ((string) ($existing->banner ?? '') === '') {
                $update['banner'] = $this->ensureGameBanner($brand);
            }

            $existing->update($update);
            return $existing;
        }

        return Game::create([
            'brand' => $brand,
            'image' => $this->ensureGameImage($brand),
            'banner' => $this->ensureGameBanner($brand),
            'title' => $cleanTitle !== '' ? $cleanTitle : $brand,
            'developers' => $developers !== '' ? $developers : ($brand !== '' ? $brand : 'Developer'),
            'category_id' => (string) $category->id,
            'slug' => Str::slug($brand),
            'description' => $desc,
            'status' => true,
            'populer' => false,
            'sort' => 0,
        ]);
    }

    public function fetchProduct()
    {
        $settings = Setting::all()->pluck('value', 'key');

        $this->siteName = (string) (
            $settings['site.name'] ??
            $settings['web.name'] ??
            $settings['app.name'] ??
            config('app.name') ??
            'Website'
        );

        $username = (string) ($settings['digi.username'] ?? '');
        $apiKey = (string) ($settings['digi.apikey'] ?? '');

        if ($username === '' || $apiKey === '') {
            Log::error('Failed to fetch product: Digiflazz credential not set');
            return response()->json([
                'success' => false,
                'message' => 'Digiflazz credential not set'
            ], 400);
        }

        $signature = md5($username . $apiKey . 'pricelist');

        $response = Http::post('https://api.digiflazz.com/v1/price-list', [
            'cmd' => 'prepaid',
            'username' => $username,
            'sign' => $signature,
        ]);

        if (!$response->successful()) {
            $errorMessage = $response->json()['message'] ?? 'Unknown error';
            Log::error('Failed to fetch product: ' . $errorMessage);

            return response()->json([
                'success' => false,
                'message' => $errorMessage
            ], 400);
        }

        $products = $response->json()['data'] ?? [];
        if (!is_array($products) || count($products) === 0) {
            Log::error('Failed to fetch product: empty data');
            return response()->json([
                'success' => false,
                'message' => 'Empty data from Digiflazz'
            ], 400);
        }

        $syncSkuCodes = [];
        foreach ($products as $p) {
            $c = trim((string) ($p['buyer_sku_code'] ?? ''));
            if ($c !== '') $syncSkuCodes[] = $c;
        }
        $syncSkuCodes = array_values(array_unique($syncSkuCodes));
        if (count($syncSkuCodes) === 0) {
            Log::error('Failed to fetch product: empty sku list');
            return response()->json([
                'success' => false,
                'message' => 'Empty sku list from Digiflazz'
            ], 400);
        }

        $result = [
            'inserted' => 0,
            'updated' => 0,
            'deleted' => 0,
        ];

        try {
            DB::transaction(function () use ($products, $settings, $syncSkuCodes, &$result) {
                foreach ($products as $product) {
                    $productCode = trim((string) ($product['buyer_sku_code'] ?? ''));
                    if ($productCode === '') continue;

                    $productName = trim((string) ($product['product_name'] ?? ''));
                    $productBrand = trim((string) ($product['brand'] ?? ''));

                    $categoryTitle = trim((string) ($product['type'] ?? ''));
                    if ($categoryTitle === '') $categoryTitle = 'Umum';

                    $categoryGame = $this->normalizeDigiflazzMainCategory($product['category'] ?? null);

                    $isPB = $this->isPointBlank($productBrand) || $this->isPointBlank($productName);
                    $isVoucherPB = $isPB && $this->isVoucherProductName($productName);

                    if ($isPB) {
                        $categoryGame = $isVoucherPB ? 'Voucher' : 'Games';
                    }

                    $mainCategory = $this->ensureMainCategory($categoryGame);

                    if ($isPB) {
                        $gameBrand = $isVoucherPB ? 'POINT BLANK VOUCHER' : 'POINT BLANK';
                        $gameTitle = $isVoucherPB ? 'Point Blank Voucher' : 'Point Blank';
                    } else {
                        $gameBrand = $productBrand !== '' ? $productBrand : $categoryTitle;
                        $gameTitle = $productBrand !== '' ? $productBrand : $categoryTitle;
                    }

                    $game = $this->ensureGameFromDigiflazz($gameBrand, $gameTitle, $mainCategory);

                    $profitRecord = $game ? Profit::where('game_id', $game->id)->first() : null;

                    $profitPercentage = $profitRecord ? (float) $profitRecord->profit_percentage : (float) ($settings['digi.profit'] ?? 0);
                    $profitGoldPercentage = $profitRecord ? (float) $profitRecord->profit_gold_percentage : (float) ($settings['digi.profit_gold'] ?? 0);
                    $profitPlatinumPercentage = $profitRecord ? (float) $profitRecord->profit_platinum_percentage : (float) ($settings['digi.profit_platinum'] ?? 0);

                    $price = (float) ($product['price'] ?? 0);

                    $profit = ($price * $profitPercentage) / 100;
                    $sellingPrice = $price + $profit;

                    $profitGold = ($price * $profitGoldPercentage) / 100;
                    $sellingPriceGold = $price + $profitGold;

                    $profitPlatinum = ($price * $profitPlatinumPercentage) / 100;
                    $sellingPricePlatinum = $price + $profitPlatinum;

                    $logoPath = $this->ensureCategoryLogo($categoryTitle, $categoryGame);
                    $bannerPath = $this->ensureProductBanner($productName !== '' ? $productName : $productCode, $productBrand, $productCode);

                    $category = ProductCategory::firstOrCreate(
                        ['title' => $categoryTitle, 'game' => $categoryGame],
                        ['logo' => $logoPath]
                    );

                    if ((string) ($category->logo ?? '') === '') {
                        $category->update(['logo' => $logoPath]);
                    }

                    $rawProductTitle = $productName !== '' ? $productName : $productCode;

                    if ($isPB && $isVoucherPB) {
                        $rawClean = $this->smartTitle($rawProductTitle);
                        $rawClean = trim(preg_replace('/^point\s*blank\s*[-–—|:]\s*/iu', '', $rawClean) ?? $rawClean);
                        $rawClean = trim(preg_replace('/^point\s*blank\s+/iu', '', $rawClean) ?? $rawClean);

                        $t = trim($rawClean);
                        $t = preg_replace('/^voucher\s*/iu', '', $t) ?? $t;
                        $t = trim($t);

                        $cleanProductTitle = 'Voucher ' . ($t !== '' ? $t : $rawProductTitle);
                    } else {
                        $gameDisplayName = $this->smartTitle($gameTitle !== '' ? $gameTitle : $gameBrand);
                        $withoutGameName = $this->stripLeadingGameName($rawProductTitle, $gameDisplayName, $gameBrand);
                        $cleanProductTitle = $this->smartTitle($withoutGameName !== '' ? $withoutGameName : $rawProductTitle);
                    }

                    $gameDisplayName = $this->smartTitle($gameTitle !== '' ? $gameTitle : $gameBrand);
                    $desc = $this->descriptionFor($gameDisplayName !== '' ? $gameDisplayName : $gameBrand);

                    $existing = Product::where('code', $productCode)->first();

                    $payload = [
                        'title' => $cleanProductTitle !== '' ? $cleanProductTitle : $productCode,
                        'brand' => $productBrand !== '' ? $productBrand : $gameBrand,
                        'type' => (string) ($product['type'] ?? '') !== '' ? (string) ($product['type'] ?? '') : 'Umum',
                        'category' => (string) $category->id,
                        'type_transaction' => 'otomatis',
                        'seller_name' => (string) ($product['seller_name'] ?? '') !== '' ? (string) ($product['seller_name'] ?? '') : 'Digiflazz',
                        'seller_price' => $price,
                        'profit' => $profit,
                        'profit_gold' => $profitGold,
                        'profit_platinum' => $profitPlatinum,
                        'selling_price' => $sellingPrice,
                        'selling_price_gold' => $sellingPriceGold,
                        'selling_price_platinum' => $sellingPricePlatinum,
                        'description' => $desc,
                        'provider' => 'Digiflazz',
                        'buyer_product_status' => (bool) ($product['buyer_product_status'] ?? true),
                        'seller_product_status' => (bool) ($product['seller_product_status'] ?? true),
                        'unlimited_stock' => (bool) ($product['unlimited_stock'] ?? true),
                        'stock' => (int) ($product['stock'] ?? 0),
                        'multi' => (bool) ($product['multi'] ?? false),
                        'start_cut_off' => (string) ($product['start_cut_off'] ?? '') !== '' ? (string) ($product['start_cut_off'] ?? '') : '-',
                        'end_cut_off' => (string) ($product['end_cut_off'] ?? '') !== '' ? (string) ($product['end_cut_off'] ?? '') : '-',
                        'status' => (bool) ($product['buyer_product_status'] ?? true) && (bool) ($product['seller_product_status'] ?? true),
                    ];

                    if ($existing) {
                        if ((string) ($existing->images ?? '') === '') {
                            $payload['images'] = $logoPath;
                        }
                        if ((string) ($existing->banner ?? '') === '') {
                            $payload['banner'] = $bannerPath;
                        }

                        $existing->update($payload);
                        $result['updated']++;
                    } else {
                        $payload['code'] = $productCode;
                        $payload['images'] = $logoPath;
                        $payload['banner'] = $bannerPath;

                        Product::create($payload);
                        $result['inserted']++;
                    }
                }

                $result['deleted'] = 0;
            });
        } catch (\Throwable $e) {
            Log::error('Failed to fetch product: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => $e->getMessage(),
            ], 400);
        }

        return response()->json([
            'success' => true,
            'message' => 'Product updated successfully.',
            'result' => $result,
        ]);
    }
}