<?php

namespace App\Services;

use App\Models\Order;
use App\Models\Setting;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class DigiflazzService
{
    protected string $username;
    protected string $apiKey;
    protected string $apiUrl;

    public function __construct()
    {
        $settings = Setting::query()->pluck('value', 'key')->all();

        $this->username = $this->normalizeSettingString((string) ($settings['digi.username'] ?? ''));
        $this->apiKey = $this->normalizeSettingString((string) ($settings['digi.apikey'] ?? ''));
        $this->apiUrl = 'https://api.digiflazz.com/v1/transaction';
    }

    public function submit(Order $order): array
    {
        return $this->processDigiflazzTransaction($order);
    }

    public function processDigiflazzTransaction(Order $order): array
    {
        if ($this->username === '' || $this->apiKey === '') {
            $msg = 'Digiflazz credential belum di-set (digi.username / digi.apikey).';
            $order->update([
                'buy_status' => 'Gagal',
                'df_message' => $msg,
            ]);

            return ['success' => false, 'message' => $msg];
        }

        $refId = trim((string) $order->order_id);
        $sku = trim((string) ($order->code_product ?? ''));
        $customerNo = $this->buildCustomerNo(
            trim((string) ($order->id_games ?? '')),
            trim((string) ($order->server_games ?? ''))
        );

        if ($refId === '' || $sku === '' || $customerNo === '') {
            $msg = 'Payload Digiflazz tidak lengkap (ref_id / buyer_sku_code / customer_no kosong).';
            $order->update(['df_message' => $msg]);

            return ['success' => false, 'message' => $msg];
        }

        $sign = md5($this->username . $this->apiKey . $refId);

        $payload = [
            'username' => $this->username,
            'buyer_sku_code' => $sku,
            'customer_no' => $customerNo,
            'ref_id' => $refId,
            'sign' => $sign,
            'testing' => false,
        ];

        try {
            $res = Http::withOptions(['http_errors' => false])
                ->retry(2, 600)
                ->timeout(25)
                ->acceptJson()
                ->asJson()
                ->post($this->apiUrl, $payload);

            $json = $res->json();

            if (!is_array($json)) {
                $msg = 'Response Digiflazz tidak valid.';
                $order->update(['df_message' => $msg]);

                Log::warning('[DIGIFLAZZ] invalid json', [
                    'order_id' => $refId,
                    'status' => $res->status(),
                ]);

                return ['success' => false, 'message' => $msg];
            }

            if (!$res->successful()) {
                $msg = (string) data_get($json, 'data.message', 'Gagal menghubungi Digiflazz.');
                $msg = trim($msg);

                $update = ['df_message' => $msg];

                if ($this->isSkuInvalidMessage($msg) || $this->isInsufficientBalanceMessage($msg)) {
                    $update['buy_status'] = 'Gagal';
                }

                $order->update($update);

                Log::warning('[DIGIFLAZZ] http not ok', [
                    'order_id' => $refId,
                    'status' => $res->status(),
                    'body' => $this->shortBody($json),
                ]);

                return ['success' => false, 'message' => $msg, 'raw' => $json];
            }

            $data = is_array($json['data'] ?? null) ? $json['data'] : [];
            $statusRaw = (string) ($data['status'] ?? '');
            $message = trim((string) ($data['message'] ?? ''));
            $sn = isset($data['sn']) ? trim((string) $data['sn']) : '';

            $mapped = $this->mapBuyStatus($statusRaw);

            $update = [];
            if ($mapped !== null) $update['buy_status'] = $mapped;
            if ($message !== '') $update['df_message'] = $message;

            if (($update['buy_status'] ?? $order->buy_status) === 'Sukses' && $sn !== '') {
                $update['serial_number'] = $sn;
            }

            if ($message !== '' && ($this->isSkuInvalidMessage($message) || $this->isInsufficientBalanceMessage($message))) {
                $update['buy_status'] = 'Gagal';
                $update['df_message'] = $message;
            }

            if (!empty($update)) {
                $order->update($update);
            }

            return [
                'success' => $this->isSyncOk($statusRaw),
                'message' => $message !== '' ? $message : 'OK',
                'data' => $data,
            ];
        } catch (\Throwable $e) {
            $msg = 'Terjadi kesalahan saat menghubungi Digiflazz.';
            $order->update(['df_message' => $msg]);

            Log::warning('[DIGIFLAZZ] process exception', [
                'order_id' => $refId,
                'message' => $e->getMessage(),
            ]);

            return ['success' => false, 'message' => $msg];
        }
    }

    public function checkStatus(string $refId): array
    {
        $refId = trim((string) $refId);
        if ($refId === '') {
            return [
                'success' => false,
                'message' => 'ref_id kosong.',
            ];
        }

        return [
            'success' => false,
            'message' => 'Method deprecated. Gunakan syncStatus($order).',
        ];
    }

    public function syncStatus(Order $order): array
    {
        if ($this->username === '' || $this->apiKey === '') {
            return [
                'success' => false,
                'message' => 'Digiflazz credential belum di-set (digi.username / digi.apikey).',
            ];
        }

        $refId = trim((string) $order->order_id);
        $sku = trim((string) ($order->code_product ?? ''));
        $customerNo = $this->buildCustomerNo(
            trim((string) ($order->id_games ?? '')),
            trim((string) ($order->server_games ?? ''))
        );

        if ($refId === '' || $sku === '' || $customerNo === '') {
            $msg = 'Tidak bisa sync Digiflazz: payload tidak lengkap (ref_id/sku/customer_no).';
            $order->update(['df_message' => $msg]);

            return ['success' => false, 'message' => $msg];
        }

        $sign = md5($this->username . $this->apiKey . $refId);

        $payload = [
            'cmd' => 'status',
            'username' => $this->username,
            'ref_id' => $refId,
            'buyer_sku_code' => $sku,
            'customer_no' => $customerNo,
            'sign' => $sign,
        ];

        try {
            $res = Http::withOptions(['http_errors' => false])
                ->retry(2, 600)
                ->timeout(25)
                ->acceptJson()
                ->asJson()
                ->post($this->apiUrl, $payload);

            $json = $res->json();

            if (!is_array($json)) {
                $msg = 'Response Digiflazz tidak valid.';
                $order->update(['df_message' => $msg]);

                Log::warning('[DIGIFLAZZ] checkStatus invalid json', [
                    'order_id' => $refId,
                    'status' => $res->status(),
                ]);

                return ['success' => false, 'message' => $msg];
            }

            if (!$res->successful()) {
                $msg = (string) data_get($json, 'data.message', 'Gagal mengambil status Digiflazz.');
                $msg = trim($msg);

                $update = ['df_message' => $msg];

                if ($this->isSkuInvalidMessage($msg) || $this->isInsufficientBalanceMessage($msg)) {
                    $update['buy_status'] = 'Gagal';
                }

                $order->update($update);

                Log::warning('[DIGIFLAZZ] checkStatus http not ok', [
                    'order_id' => $refId,
                    'status' => $res->status(),
                    'body' => $this->shortBody($json),
                ]);

                return ['success' => false, 'message' => $msg, 'raw' => $json];
            }

            $data = $json['data'] ?? null;
            if (!is_array($data)) {
                $msg = 'Response Digiflazz tidak valid.';
                $order->update(['df_message' => $msg]);

                Log::warning('[DIGIFLAZZ] checkStatus invalid data', [
                    'order_id' => $refId,
                    'body' => $this->shortBody($json),
                ]);

                return ['success' => false, 'message' => $msg, 'raw' => $json];
            }

            $statusRaw = (string) ($data['status'] ?? '');
            $message = trim((string) ($data['message'] ?? ''));
            $sn = isset($data['sn']) ? trim((string) $data['sn']) : '';

            $mapped = $this->mapBuyStatus($statusRaw);

            $update = [];
            if ($mapped !== null) $update['buy_status'] = $mapped;
            if ($message !== '') $update['df_message'] = $message;

            if (($update['buy_status'] ?? $order->buy_status) === 'Sukses' && $sn !== '') {
                $update['serial_number'] = $sn;
            }

            if ($message !== '' && ($this->isSkuInvalidMessage($message) || $this->isInsufficientBalanceMessage($message))) {
                $update['buy_status'] = 'Gagal';
                $update['df_message'] = $message;
            }

            if (!empty($update)) {
                $order->update($update);
            }

            if ($message !== '' && ($this->isSkuInvalidMessage($message) || $this->isInsufficientBalanceMessage($message))) {
                return ['success' => false, 'message' => $message, 'data' => $data];
            }

            return [
                'success' => $this->isSyncOk($statusRaw),
                'message' => $message !== '' ? $message : 'Order disinkronkan.',
                'data' => $data,
            ];
        } catch (\Throwable $e) {
            Log::warning('[DIGIFLAZZ] checkStatus exception', [
                'order_id' => $refId,
                'message' => $e->getMessage(),
            ]);

            return [
                'success' => false,
                'message' => 'Terjadi kesalahan saat mengambil status Digiflazz.',
            ];
        }
    }

    private function isSkuInvalidMessage(string $msg): bool
    {
        $m = strtolower($msg);
        return str_contains($m, 'sku') && (str_contains($m, 'tidak') || str_contains($m, 'non-aktif') || str_contains($m, 'non aktif'));
    }

    private function isInsufficientBalanceMessage(string $msg): bool
    {
        $m = strtolower(trim($msg));
        if ($m === '') return false;

        return str_contains($m, 'saldo tidak cukup')
            || str_contains($m, 'insufficient balance')
            || (str_contains($m, 'saldo') && str_contains($m, 'tidak cukup'));
    }

    private function isSyncOk(string $status): bool
    {
        $s = strtolower(trim($status));
        if ($s === 'pending') return true;
        if ($s === 'processing' || $s === 'proses') return true;
        if ($s === 'sukses' || $s === 'success') return true;
        return false;
    }

    private function mapBuyStatus(string $digiflazzStatus): ?string
    {
        $s = strtolower(trim($digiflazzStatus));

        return match ($s) {
            'pending' => 'Proses',
            'processing', 'proses' => 'Proses',
            'sukses', 'success' => 'Sukses',
            'gagal', 'failed' => 'Gagal',
            default => null,
        };
    }

    private function buildCustomerNo(string $idGames, string $serverGames): string
    {
        $id = trim($idGames);
        $sv = trim($serverGames);

        if ($id === '' && $sv === '') return '';
        if ($sv === '') return $id;

        return $id . $sv;
    }

    private function normalizeSettingString(string $value): string
    {
        $v = trim($value);
        if ($v === '') return '';

        $v = stripcslashes($v);
        $v = trim($v);

        if (str_starts_with($v, '"') && str_ends_with($v, '"')) {
            $v = trim($v, '"');
        }

        if (str_starts_with($v, "'") && str_ends_with($v, "'")) {
            $v = trim($v, "'");
        }

        return trim($v);
    }

    private function shortBody($json): array
    {
        if (!is_array($json)) return ['_raw' => 'non-array'];
        $data = $json;

        if (isset($data['data']) && is_array($data['data'])) {
            $data['data'] = array_intersect_key($data['data'], array_flip(['status', 'message', 'rc', 'sn', 'ref_id', 'buyer_sku_code', 'customer_no']));
        }

        return array_intersect_key($data, array_flip(['data', 'message', 'rc', 'status']));
    }
}
