<?php

namespace App\Services;

use App\Models\Game;
use App\Models\Order;
use App\Models\PaymentMethod;
use App\Models\Product;
use App\Models\PromoCode;
use Illuminate\Support\Carbon;

class PromoCodeService
{
    public function resolveBasePrice(Product $product, ?string $role): int
    {
        $selling = match ($role) {
            'platinum' => (int) ($product->selling_price_platinum ?? $product->selling_price),
            'gold' => (int) ($product->selling_price_gold ?? $product->selling_price),
            default => (int) $product->selling_price,
        };

        $final = (int) ($product->promo_price ?? $selling);

        if ($final < 0) $final = 0;

        return $final;
    }

    public function computeDiscount(PromoCode $promo, int $basePrice): int
    {
        if ($basePrice <= 0) return 0;

        $discount = 0;

        if ($promo->discount_type === 'PERCENT') {
            $percent = (int) $promo->discount_value;
            if ($percent < 1) $percent = 1;
            if ($percent > 100) $percent = 100;
            $discount = (int) floor($basePrice * ($percent / 100));
        } else {
            $discount = (int) $promo->discount_value;
        }

        if ($discount < 0) $discount = 0;
        if ($discount > $basePrice) $discount = $basePrice;

        return $discount;
    }

    public function validatePromo(
        string $code,
        string $gameSlug,
        int $productId,
        int $paymentMethodId,
        ?string $role,
        ?int $userId,
        ?string $whatsapp
    ): array {
        $promo = PromoCode::query()
            ->where('code', $code)
            ->first();

        if (!$promo) {
            return $this->invalid('Kode promo tidak ditemukan.');
        }

        $now = Carbon::now();

        if (!$promo->is_active) return $this->invalid('Kode promo tidak aktif.');
        if ($promo->start_at && $promo->start_at->gt($now)) return $this->invalid('Kode promo belum berlaku.');
        if ($promo->end_at && $promo->end_at->lt($now)) return $this->invalid('Kode promo sudah berakhir.');

        $game = Game::query()->where('slug', $gameSlug)->first();
        if (!$game) return $this->invalid('Game tidak ditemukan.');

        $product = Product::query()->whereKey($productId)->first();
        if (!$product) return $this->invalid('Produk tidak ditemukan.');

        $payment = PaymentMethod::query()->whereKey($paymentMethodId)->first();
        if (!$payment) return $this->invalid('Metode pembayaran tidak ditemukan.');
        if (!(bool) $payment->status) return $this->invalid('Metode pembayaran tidak aktif.');

        $restrictedPayment = $promo->paymentMethods()->exists();
        if ($restrictedPayment) {
            $allowed = $promo->paymentMethods()->whereKey($paymentMethodId)->exists();
            if (!$allowed) return $this->invalid('Kode promo tidak berlaku untuk metode pembayaran ini.');
        }

        $restrictedGame = $promo->games()->exists();
        if ($restrictedGame) {
            $allowedGame = $promo->games()->whereKey($game->id)->exists();
            if (!$allowedGame) return $this->invalid('Kode promo tidak berlaku untuk game ini.');
        }

        $basePrice = $this->resolveBasePrice($product, $role);

        if ($promo->min_product_price !== null) {
            $min = (int) $promo->min_product_price;
            if ($basePrice < $min) return $this->invalid('Kode promo tidak berlaku untuk harga produk ini.');
        }

        if ($promo->usage_limit_total !== null) {
            $limit = (int) $promo->usage_limit_total;
            if ((int) $promo->used_count >= $limit) return $this->invalid('Kuota penggunaan kode promo sudah habis.');
        }

        if ($promo->usage_limit_per_user !== null) {
            $limitUser = (int) $promo->usage_limit_per_user;
            $usedByUser = $this->countUsageByUser($promo->id, $userId, $whatsapp);
            if ($usedByUser >= $limitUser) return $this->invalid('Batas penggunaan kode promo untuk akun ini sudah tercapai.');
        }

        $discount = $this->computeDiscount($promo, $basePrice);
        $final = $basePrice - $discount;
        if ($final < 0) $final = 0;

        return [
            'valid' => true,
            'message' => 'Kode promo dapat digunakan.',
            'promo' => [
                'id' => $promo->id,
                'code' => $promo->code,
                'name' => $promo->name,
                'discount_type' => $promo->discount_type,
                'discount_value' => (int) $promo->discount_value,
                'min_product_price' => $promo->min_product_price !== null ? (int) $promo->min_product_price : null,
                'start_at' => $promo->start_at?->toIso8601String(),
                'end_at' => $promo->end_at?->toIso8601String(),
            ],
            'pricing' => [
                'base_price' => $basePrice,
                'discount' => $discount,
                'final_price' => $final,
            ],
        ];
    }

    public function listForPopup(?string $gameSlug, ?int $paymentMethodId, ?int $productId, ?string $role): array
    {
        $now = Carbon::now();

        $game = $gameSlug ? Game::query()->where('slug', $gameSlug)->first() : null;
        $product = $productId ? Product::query()->whereKey($productId)->first() : null;
        $payment = $paymentMethodId ? PaymentMethod::query()->whereKey($paymentMethodId)->first() : null;

        $basePrice = $product ? $this->resolveBasePrice($product, $role) : null;

        $promos = PromoCode::query()
            ->orderByDesc('id')
            ->limit(100)
            ->get()
            ->map(function (PromoCode $promo) use ($now, $game, $payment, $basePrice, $gameSlug, $paymentMethodId, $productId) {
                $status = 'ACTIVE';

                if (!$promo->is_active) $status = 'INACTIVE';
                if ($promo->start_at && $promo->start_at->gt($now)) $status = 'UPCOMING';
                if ($promo->end_at && $promo->end_at->lt($now)) $status = 'EXPIRED';

                $eligible = $status === 'ACTIVE';
                $reasons = [];

                $restrictedPayment = $promo->paymentMethods()->exists();
                if ($eligible && $restrictedPayment) {
                    if (!$payment) {
                        $eligible = false;
                        $reasons[] = 'PAYMENT_NOT_FOUND';
                    } else {
                        $allowed = $promo->paymentMethods()->whereKey($payment->id)->exists();
                        if (!$allowed) {
                            $eligible = false;
                            $reasons[] = 'PAYMENT_NOT_ALLOWED';
                        }
                    }
                }

                $restrictedGame = $promo->games()->exists();
                if ($eligible && $restrictedGame) {
                    if (!$game) {
                        $eligible = false;
                        $reasons[] = 'GAME_NOT_FOUND';
                    } else {
                        $allowedGame = $promo->games()->whereKey($game->id)->exists();
                        if (!$allowedGame) {
                            $eligible = false;
                            $reasons[] = 'GAME_NOT_ALLOWED';
                        }
                    }
                }

                if ($eligible && $promo->min_product_price !== null) {
                    if ($basePrice === null) {
                        $eligible = false;
                        $reasons[] = 'PRODUCT_NOT_FOUND';
                    } else {
                        if ($basePrice < (int) $promo->min_product_price) {
                            $eligible = false;
                            $reasons[] = 'MIN_PRICE_NOT_MET';
                        }
                    }
                }

                if ($eligible && $promo->usage_limit_total !== null) {
                    if ((int) $promo->used_count >= (int) $promo->usage_limit_total) {
                        $eligible = false;
                        $reasons[] = 'QUOTA_EXCEEDED';
                    }
                }

                $debug = [
                    'input' => [
                        'game_slug' => $gameSlug,
                        'payment_method_id' => $paymentMethodId,
                        'product_id' => $productId,
                    ],
                    'resolved' => [
                        'game_id' => $game?->id,
                        'payment_id' => $payment?->id,
                        'base_price' => $basePrice,
                        'restricted_payment' => (bool) $restrictedPayment,
                        'restricted_game' => (bool) $restrictedGame,
                    ],
                ];

                return [
                    'id' => $promo->id,
                    'name' => $promo->name,
                    'code' => $promo->code,
                    'status' => $status,
                    'is_eligible' => (bool) $eligible,
                    'ineligible_reasons' => $eligible ? [] : $reasons,
                    'discount_type' => $promo->discount_type,
                    'discount_value' => (int) $promo->discount_value,
                    'min_product_price' => $promo->min_product_price !== null ? (int) $promo->min_product_price : null,
                    'start_at' => $promo->start_at?->toIso8601String(),
                    'end_at' => $promo->end_at?->toIso8601String(),
                    'usage_limit_total' => $promo->usage_limit_total !== null ? (int) $promo->usage_limit_total : null,
                    'used_count' => (int) $promo->used_count,
                    'debug' => $debug,
                ];
            })
            ->values();

        return [
            'success' => true,
            'data' => $promos,
        ];
    }

    private function countUsageByUser(int $promoId, ?int $userId, ?string $whatsapp): int
    {
        $q = Order::query()->where('promo_code_id', $promoId);

        if ($userId) {
            $q->where('user_id', $userId);
            return (int) $q->count();
        }

        if ($whatsapp) {
            $q->where('whatsapp', $whatsapp);
            return (int) $q->count();
        }

        return 0;
    }

    private function invalid(string $message): array
    {
        return [
            'valid' => false,
            'message' => $message,
        ];
    }
}