<?php

namespace App\Services;

use App\Models\Setting;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class WhatsAppNotificationService
{
    public function send(string $phone, string $templateKey, array $vars = [], ?string $fallbackMessage = null): void
    {
        $settings = Setting::all()->pluck('value', 'key');

        $enabled = $settings["notification.enabled.$templateKey"] ?? true;

        if ($enabled === false || $enabled === '0' || $enabled === 0) {
            return;
        }

        $vars = $this->normalizeVars($vars);

        $provider = $settings['notification.provider'] ?? 'fonnte';
        $template = $settings["notification.templates.$templateKey"] ?? null;

        $message = $this->render($template, $vars) ?: $fallbackMessage;

        if (! $message) {
            return;
        }

        if ($provider === 'mpwa') {
            $this->sendMpwa($settings, $phone, $message);
            return;
        }

        $this->sendFonnte($settings, $phone, $message);
    }

    private function normalizeVars(array $vars): array
    {
        if (isset($vars['expired_time']) && $this->isNumericValue($vars['expired_time'])) {
            $vars['expired_time'] = $this->formatWibDateTime((int) $this->toInt($vars['expired_time']));
        }

        foreach (['price', 'fee', 'total_price'] as $key) {
            if (isset($vars[$key]) && $this->isNumericValue($vars[$key])) {
                $vars[$key] = $this->formatRupiah($this->toInt($vars[$key]));
            }
        }

        return $vars;
    }

    private function formatRupiah(int $value): string
    {
        return 'Rp ' . number_format($value, 0, ',', '.');
    }

    private function formatWibDateTime(int $timestamp): string
    {
        return date('d M Y, H:i', $timestamp) . ' WIB';
    }

    private function isNumericValue($value): bool
    {
        if (is_int($value) || is_float($value)) {
            return true;
        }

        if (! is_string($value)) {
            return false;
        }

        $digits = preg_replace('/\D+/', '', $value) ?? '';
        return $digits !== '';
    }

    private function toInt($value): int
    {
        if (is_int($value)) {
            return $value;
        }

        if (is_float($value)) {
            return (int) round($value);
        }

        $digits = preg_replace('/\D+/', '', (string) $value) ?? '0';
        return (int) $digits;
    }

    private function render($template, array $vars): ?string
    {
        $template = is_string($template) ? trim($template) : null;

        if (! $template) {
            return null;
        }

        $replacements = [];
        foreach ($vars as $key => $value) {
            $replacements['{' . $key . '}'] = is_scalar($value) || $value === null ? (string) $value : json_encode($value);
        }

        return strtr($template, $replacements);
    }

    private function normalizePhone(string $phone): string
    {
        $digits = preg_replace('/\D+/', '', $phone) ?? '';

        if (str_starts_with($digits, '0')) {
            return '62' . substr($digits, 1);
        }

        if (str_starts_with($digits, '8')) {
            return '62' . $digits;
        }

        return $digits;
    }

    private function sendFonnte($settings, string $phone, string $message): void
    {
        $token = $settings['fonnte.token'] ?? null;

        if (! $token) {
            return;
        }

        $response = Http::withHeaders([
            'Authorization' => $token,
        ])->asForm()->post('https://api.fonnte.com/send', [
            'target' => $this->normalizePhone($phone),
            'message' => $message,
            'countryCode' => '62',
        ]);

        if (! $response->successful()) {
            Log::error('Gagal kirim WhatsApp Fonnte: ' . $response->body());
        }
    }

    private function sendMpwa($settings, string $phone, string $message): void
    {
        $apiKey = $settings['mpwa.api_key'] ?? null;
        $sender = $settings['mpwa.sender'] ?? null;

        if (! $apiKey || ! $sender) {
            return;
        }

        $footer = $settings['mpwa.footer'] ?? null;

        $result = app(MpwaService::class)->sendMessage(
            $apiKey,
            $sender,
            $this->normalizePhone($phone),
            $message,
            is_string($footer) && trim($footer) ? $footer : null
        );

        if (! data_get($result, 'status')) {
            Log::error('Gagal kirim WhatsApp MPWA: ' . json_encode($result));
        }
    }
}